
#include "Mesquite_all_headers.hpp"
#include <iostream>
#include <sstream>

using std::cout;
using std::cerr;
using std::endl;
using std::ostream;
using std::ostringstream;

#include <memory>
using std::auto_ptr;

#include <ctype.h>

using namespace MESQUITE_NS;


void deformBoundary( MeshImpl& deformedMesh, MeshImpl& origMesh, double a)
{
  MsqPrintError err(cerr);

  std::vector<Mesh::VertexHandle> vertices;
  origMesh.get_all_vertices(vertices, err);
  if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}
  size_t num_vertices = vertices.size();
  
  cout<<"num vertices="<<num_vertices<<endl;
  // get vertex coordinates
  std::vector<MsqVertex> coordinates(num_vertices);
  origMesh.vertices_get_coordinates( arrptr(vertices), arrptr(coordinates), num_vertices, err);
  if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}
  
  // get application fixed vertices
  std::vector<bool> app_fixed(num_vertices);
  origMesh.vertices_get_fixed_flag( arrptr(vertices), app_fixed, num_vertices, err );

  if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}
  
  cout<<"num fixed vertices="<<app_fixed.size()<<endl;

  double twopi=8.0*atan(1.0);

  for(size_t i=0;i<num_vertices; i++)
  {
    if( app_fixed[i] == true )
    {
      double x = coordinates[i].x();
      double y = coordinates[i].y();

      if( ( (1-(x*x/4.0 + y*y/16.0))<.00001 ) && (y>.01) )
      {
	//
	// set new boundary vertices for deformed mesh
	//
	Vector3D newPos(x,a*y,0.0);
	deformedMesh.vertex_set_coordinates(vertices[i], newPos, err);
	if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}

      }
    }
  }//end of loop over vertices

  deformedMesh.write_vtk("DeformedMesh.vtk", err);
  if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}

  cout<<"Leaving deformBoundary."<<endl;
}//end of deformBoundary




int do_smoother( const char* input_file, 
                 const char* output_file,
		 double a)
{
  MsqPrintError err(cerr);
   
  //
  // Mesh we will be optimizing
  //
  MeshImpl mesh;
  mesh.read_vtk( input_file, err );

  mesh.write_vtk( "InitialMesh.vtk", err );
  if (MSQ_CHKERR(err)) return 2;

  MeshImpl deformedMesh;
  deformedMesh.read_vtk( input_file, err );

  deformBoundary(deformedMesh, mesh, a);
  //
  // constrain it to be planar.
  //
  if (MSQ_CHKERR(err)) return 2;
  XYPlanarDomain geom;
  if (MSQ_CHKERR(err)) return 1;

  TShapeSizeB1 target_metric;

  // IdealShapeTarget W;

  ReferenceMesh refMesh( &mesh );
  RefMeshTargetCalculator W( &refMesh, false );

  TQualityMetric mu( &W, &target_metric );
  
  TerminationCriterion outer, inner;
  
  PMeanPTemplate objective_function( 2.0, &mu);
  
  FeasibleNewton solver( &objective_function );
  if (MSQ_CHKERR(err)) return 1;

  solver.set_inner_termination_criterion( &inner );
  solver.set_outer_termination_criterion( &outer );
  
  bool isSolverGlobal = true;
  //bool isSolverGlobal = false;
  if( isSolverGlobal )
  {
    solver.use_global_patch();

    outer.add_iteration_limit( 1 );
    //       outer.add_relative_successive_improvement(1e-5);

    //       inner.add_absolute_vertex_movement(1e-5);
    inner.add_relative_successive_improvement(1e-5);
    // inner.write_mesh_steps("mesh_");
    //       inner.add_relative_element_overlap(0.5);

  }
  else  // local solver
  {
    solver.use_element_on_vertex_patch();

    outer.add_relative_successive_improvement(1e-6);
    //       outer.add_iteration_limit( 10 );
    // outer.write_mesh_steps("mesh_");

    //   inner.add_absolute_vertex_movement(1e-5);
    inner.add_relative_successive_improvement(1e-5);
    // inner.add_iteration_limit( 2 );
  }
  
  
  ConditionNumberQualityMetric qm_metric;
  QualityAssessor assessor;
  assessor.add_quality_assessment( &mu, 10 );
  assessor.add_quality_assessment( &qm_metric );

  InstructionQueue q;
  q.add_quality_assessor( &assessor, err );
  q.set_master_quality_improver( &solver, err );
  q.add_quality_assessor( &assessor, err );

#if 0
  Settings theSettings;
  theSettings.set_slaved_ho_node_mode(Settings::SLAVE_NONE);
	
  q.run_common( &deformedMesh, NULL, &geom, &theSettings, err );
#else
  q.run_instructions( &deformedMesh, &geom, err);
#endif
  if (MSQ_CHKERR(err)) return 3;

  deformedMesh.write_vtk( output_file, err );
  if (MSQ_CHKERR(err)) return 2;
  cout << "Wrote: " << output_file << endl;
  cout<<" Convergence in "<<outer.get_iteration_count()<<" outer iterations."<<endl;
  cout<<" Convergence in "<<inner.get_iteration_count()<<" inner iterations."<<endl;

  cout<<"Leaving do_smoother."<<endl;
  return 0;
}

int main( int argc, char* argv[] )
{
  MsqPrintError err(cout);
    
  ostringstream meshfile,outfile;
  double a=1.1;

  if( argc > 1 )
  {
    a = atof(argv[1]);
  }
    
  meshfile <<"/g/g92/kim74/hb_initial.vtk";
    
  outfile<<"SmoothedMesh.vtk";
  int result = do_smoother( meshfile.str().c_str(), outfile.str().c_str(), a);

  cout<<" main finished."<<endl;

  return result;
}

